import type { NextApiRequest, NextApiResponse } from "next";
import { prisma } from "@/lib/prisma";
import { requireAdmin } from "@/lib/admin";
import { getCurrentPeriodDate, isPeriodInPast } from "@/lib/taskPeriods";

export default async function handler(
  req: NextApiRequest,
  res: NextApiResponse
) {
  const session = await requireAdmin(req, res);
  if (!session) return;

  if (req.method !== "GET") {
    return res.status(405).json({ ok: false, error: "Method not allowed" });
  }

  try {
    // Get all tasks with related data
    const tasks = await prisma.task.findMany({
      include: {
        createdBy: {
          select: {
            id: true,
            name: true,
            email: true,
          },
        },
        assignedTo: {
          select: {
            id: true,
            name: true,
            email: true,
            isExternal: true,
            companyName: true,
          },
        },
        department: {
          select: {
            id: true,
            name: true,
          },
        },
        location: {
          select: {
            id: true,
            name: true,
          },
        },
        element: {
          select: {
            id: true,
            name: true,
          },
        },
        periodCompletions: {
          select: {
            id: true,
            periodDate: true,
            completedAt: true,
            submittedAt: true,
            completedBy: {
              select: {
                id: true,
                name: true,
                email: true,
                isExternal: true,
                companyName: true,
              },
            },
          },
        },
        itemCompletions: {
          select: {
            id: true,
            status: true,
            checkedAt: true,
            periodDate: true,
            element: {
              select: {
                id: true,
                name: true,
              },
            },
          },
        },
      },
    });

    const now = new Date();

    // Calculate basic statistics
    const totalTasks = tasks.length;
    const recurringTasks = tasks.filter((t) => t.isRecurring).length;
    const nonRecurringTasks = totalTasks - recurringTasks;

    // For recurring tasks, check current period completion
    // For non-recurring tasks, check completedAt
    const completedTasks = tasks.filter((task) => {
      if (task.isRecurring && task.recurrencePattern) {
        const currentPeriod = getCurrentPeriodDate(task.recurrencePattern);
        const periodCompletion = task.periodCompletions.find(
          (pc) => pc.periodDate.getTime() === currentPeriod.getTime()
        );
        return !!periodCompletion?.completedAt;
      }
      return !!task.completedAt;
    }).length;

    const pendingTasks = totalTasks - completedTasks;

    // Calculate overdue tasks
    const overdueTasks = tasks.filter((task) => {
      if (task.isRecurring && task.recurrencePattern) {
        // Check if current period is overdue
        const currentPeriod = getCurrentPeriodDate(task.recurrencePattern);
        const periodCompletion = task.periodCompletions.find(
          (pc) => pc.periodDate.getTime() === currentPeriod.getTime()
        );
        if (periodCompletion?.completedAt) return false;
        
        // Check if period is in the past
        return isPeriodInPast(currentPeriod, task.recurrencePattern);
      }
      // Non-recurring: check dueDate
      if (!task.dueDate) return false;
      return new Date(task.dueDate) < now && !task.completedAt;
    }).length;

    // Recurrence pattern statistics
    const patternStats = {
      daily: { total: 0, completed: 0 },
      weekly: { total: 0, completed: 0 },
      monthly: { total: 0, completed: 0 },
      semestral: { total: 0, completed: 0 },
    };

    tasks.forEach((task) => {
      if (task.isRecurring && task.recurrencePattern) {
        const pattern = task.recurrencePattern as keyof typeof patternStats;
        if (patternStats[pattern]) {
          patternStats[pattern].total++;
          const currentPeriod = getCurrentPeriodDate(task.recurrencePattern);
          const periodCompletion = task.periodCompletions.find(
            (pc) => pc.periodDate.getTime() === currentPeriod.getTime()
          );
          if (periodCompletion?.completedAt) {
            patternStats[pattern].completed++;
          }
        }
      }
    });

    // Department statistics
    const departmentMap = new Map<
      string,
      {
        total: number;
        completed: number;
        overdue: number;
        nonConformities: number;
      }
    >();

    tasks.forEach((task) => {
      const deptName = task.department?.name || "General";
      if (!departmentMap.has(deptName)) {
        departmentMap.set(deptName, {
          total: 0,
          completed: 0,
          overdue: 0,
          nonConformities: 0,
        });
      }
      const deptData = departmentMap.get(deptName)!;
      deptData.total++;

      // Check completion
      let isCompleted = false;
      if (task.isRecurring && task.recurrencePattern) {
        const currentPeriod = getCurrentPeriodDate(task.recurrencePattern);
        const periodCompletion = task.periodCompletions.find(
          (pc) => pc.periodDate.getTime() === currentPeriod.getTime()
        );
        isCompleted = !!periodCompletion?.completedAt;
      } else {
        isCompleted = !!task.completedAt;
      }

      if (isCompleted) {
        deptData.completed++;
      }

      // Check overdue
      let isOverdue = false;
      if (task.isRecurring && task.recurrencePattern) {
        const currentPeriod = getCurrentPeriodDate(task.recurrencePattern);
        const periodCompletion = task.periodCompletions.find(
          (pc) => pc.periodDate.getTime() === currentPeriod.getTime()
        );
        if (!periodCompletion?.completedAt) {
          isOverdue = isPeriodInPast(currentPeriod, task.recurrencePattern);
        }
      } else {
        if (task.dueDate && !task.completedAt) {
          isOverdue = new Date(task.dueDate) < now;
        }
      }

      if (isOverdue) {
        deptData.overdue++;
      }

      // Count non-conformities (NOT_OK items)
      const notOkItems = task.itemCompletions.filter(
        (ic) => ic.status === "NOT_OK"
      ).length;
      deptData.nonConformities += notOkItems;
    });

    // Technician statistics
    const technicianMap = new Map<
      string,
      {
        technicianId: string;
        technicianName: string;
        technicianEmail: string;
        isExternal: boolean;
        companyName: string | null;
        total: number;
        completed: number;
        overdue: number;
        avgCompletionTime: number;
        completionTimes: number[];
      }
    >();

    tasks.forEach((task) => {
      if (!task.assignedTo) return;

      const techId = task.assignedTo.id;
      if (!technicianMap.has(techId)) {
        technicianMap.set(techId, {
          technicianId: techId,
          technicianName: task.assignedTo.name || task.assignedTo.email,
          technicianEmail: task.assignedTo.email,
          isExternal: task.assignedTo.isExternal || false,
          companyName: task.assignedTo.companyName,
          total: 0,
          completed: 0,
          overdue: 0,
          avgCompletionTime: 0,
          completionTimes: [],
        });
      }
      const techData = technicianMap.get(techId)!;
      techData.total++;

      // Check completion
      let isCompleted = false;
      let completedAt: Date | null = null;
      if (task.isRecurring && task.recurrencePattern) {
        const currentPeriod = getCurrentPeriodDate(task.recurrencePattern);
        const periodCompletion = task.periodCompletions.find(
          (pc) => pc.periodDate.getTime() === currentPeriod.getTime()
        );
        isCompleted = !!periodCompletion?.completedAt;
        completedAt = periodCompletion?.completedAt || null;
      } else {
        isCompleted = !!task.completedAt;
        completedAt = task.completedAt;
      }

      if (isCompleted && completedAt && task.createdAt) {
        techData.completed++;
        const completionTime =
          (completedAt.getTime() - task.createdAt.getTime()) /
          (1000 * 60 * 60 * 24); // Days
        techData.completionTimes.push(completionTime);
      }

      // Check overdue
      let isOverdue = false;
      if (task.isRecurring && task.recurrencePattern) {
        const currentPeriod = getCurrentPeriodDate(task.recurrencePattern);
        const periodCompletion = task.periodCompletions.find(
          (pc) => pc.periodDate.getTime() === currentPeriod.getTime()
        );
        if (!periodCompletion?.completedAt) {
          isOverdue = isPeriodInPast(currentPeriod, task.recurrencePattern);
        }
      } else {
        if (task.dueDate && !task.completedAt) {
          isOverdue = new Date(task.dueDate) < now;
        }
      }

      if (isOverdue) {
        techData.overdue++;
      }
    });

    // Calculate average completion times
    technicianMap.forEach((techData) => {
      if (techData.completionTimes.length > 0) {
        techData.avgCompletionTime =
          techData.completionTimes.reduce((a, b) => a + b, 0) /
          techData.completionTimes.length;
      }
    });

    // Non-conformity statistics
    const totalNonConformities = tasks.reduce(
      (sum, task) =>
        sum + task.itemCompletions.filter((ic) => ic.status === "NOT_OK").length,
      0
    );

    const tasksWithNonConformities = tasks.filter(
      (task) => task.itemCompletions.some((ic) => ic.status === "NOT_OK")
    ).length;

    // Period completion statistics for recurring tasks
    const totalPeriods = tasks.reduce(
      (sum, task) => (task.isRecurring ? sum + task.periodCompletions.length : sum),
      0
    );

    const completedPeriods = tasks.reduce(
      (sum, task) =>
        task.isRecurring
          ? sum +
            task.periodCompletions.filter((pc) => !!pc.completedAt).length
          : sum,
      0
    );

    const periodCompletionRate =
      totalPeriods > 0 ? (completedPeriods / totalPeriods) * 100 : 0;

    // Time-based statistics
    const today = new Date();
    today.setHours(0, 0, 0, 0);
    const weekAgo = new Date(today);
    weekAgo.setDate(weekAgo.getDate() - 7);
    const monthAgo = new Date(today);
    monthAgo.setMonth(monthAgo.getMonth() - 1);

    const tasksCreatedToday = tasks.filter(
      (t) => new Date(t.createdAt) >= today
    ).length;

    const tasksCreatedThisWeek = tasks.filter(
      (t) => new Date(t.createdAt) >= weekAgo
    ).length;

    const tasksCreatedThisMonth = tasks.filter(
      (t) => new Date(t.createdAt) >= monthAgo
    ).length;

    res.status(200).json({
      ok: true,
      taskStats: {
        totalTasks,
        completedTasks,
        pendingTasks,
        overdueTasks,
        recurringTasks,
        nonRecurringTasks,
        periodCompletionRate,
        totalPeriods,
        completedPeriods,
        tasksCreatedToday,
        tasksCreatedThisWeek,
        tasksCreatedThisMonth,
      },
      patternStats: Object.entries(patternStats).map(([pattern, stats]) => ({
        pattern,
        total: stats.total,
        completed: stats.completed,
        completionRate:
          stats.total > 0 ? (stats.completed / stats.total) * 100 : 0,
      })),
      departmentStats: Array.from(departmentMap.entries()).map(
        ([department, stats]) => ({
          department,
          total: stats.total,
          completed: stats.completed,
          overdue: stats.overdue,
          nonConformities: stats.nonConformities,
          completionRate:
            stats.total > 0 ? (stats.completed / stats.total) * 100 : 0,
        })
      ),
      technicianStats: Array.from(technicianMap.values()).map((tech) => ({
        ...tech,
        completionRate:
          tech.total > 0 ? (tech.completed / tech.total) * 100 : 0,
      })),
      nonConformityStats: {
        totalNonConformities,
        tasksWithNonConformities,
        nonConformityRate:
          totalTasks > 0 ? (tasksWithNonConformities / totalTasks) * 100 : 0,
      },
    });
  } catch (error) {
    console.error("Error analyzing tasks:", error);
    return res.status(500).json({
      ok: false,
      error: "Failed to analyze tasks",
    });
  }
}

